#ifndef TT_INCLUDE_RAWFILECLASS_H
#define TT_INCLUDE_RAWFILECLASS_H



#include "fileclass.h"
#include "engine_math.h"
#include "engine_string.h"

class RawFileClass : public FileClass {
	int Rights; // 0004
	int BiasStart; // 0008
	int BiasLength; // 000C
	void* Handle; // 0010
	StringClass Filename; // 0014
	unsigned short Date; // 0018
	unsigned short Time; // 001C
public:
	virtual bool Is_Hash_Checked() const override
	{
		return Is_Biased();
	}

	bool Is_Biased() const
	{
		return BiasLength != -1;
	}

	RawFileClass()
	{
		BiasLength = -1;
		Handle = INVALID_HANDLE_VALUE;
		Rights = 0;
		BiasStart = 0;
		Date = 0;
		Time = 0;
		Filename = "";
	}

	RawFileClass(const char* filename) : Filename(filename)
	{
		BiasLength = -1;
		Handle = INVALID_HANDLE_VALUE;
		Rights = 0;
		BiasStart = 0;
		Date = 0;
		Time = 0;
	}

	int Transfer_Block_Size()
	{
		return -11;
	}

	void Reset()
	{
		Close();
		Filename = "";
	}

	~RawFileClass()
	{
		Reset();
	}

	virtual const char * File_Name() const override
	{
		return Filename;
	}

	const char *Set_Name(const char* name)
	{
		Reset();
		if (name)
		{
			Bias(0,-1);
			Filename = name;
		}
		return nullptr;
	}

	bool Create()
	{
		Close();
		if (!Open(2))
		{
			return false;
		}
		if (BiasLength != -1)
		{
			Seek(0,0);
		}
		Close();
		return true;
	}

	bool Delete()
	{
		Close();
		if (Filename.Is_Empty())
		{
			Error(2,0,nullptr);
		}
		if (!Is_Available(0))
		{
			return false;
		}
		if (!DeleteFileA(Filename))
		{
			Error(GetLastError(),0,Filename);
			return false;
		}
		return true;
	}

	virtual bool Is_Available(int _handle) override
	{
		if (Filename.Is_Empty())
			return false;
		
		if (Is_Open())
			return true;
		
		if (_handle)
		{
			Open(1);
			Close();
			return true;
		}

		int attr = GetFileAttributesA(Filename);
		return attr != INVALID_FILE_ATTRIBUTES && !(attr & FILE_ATTRIBUTE_DIRECTORY);
	}

	bool Is_Open() const override
	{
		if (Handle != INVALID_HANDLE_VALUE)
		{
			return true;
		}
		return false;
	}
	int Open(const char* name, int mode)
	{
		Set_Name(name);
		return Open(mode);
	}
	int Open(int mode)
	{
		Close();
		if (Filename.Is_Empty())
		{
			Error(2,0,nullptr);
		}
		Rights = mode & 3;
        DWORD flags = FILE_ATTRIBUTE_NORMAL;
        if (mode & 4) flags |= FILE_FLAG_OVERLAPPED;
		switch (mode & 3)
		{
		case 2:
			Handle = CreateFileA(Filename,GENERIC_WRITE,0,nullptr,CREATE_ALWAYS, flags,nullptr);
			break;
		case 1:
			Handle = CreateFileA(Filename,GENERIC_READ,FILE_SHARE_READ,nullptr,OPEN_EXISTING, flags,nullptr);
			break;
		case 0:
			Handle = CreateFileA(Filename,GENERIC_READ|GENERIC_WRITE,0,nullptr,OPEN_ALWAYS, flags,nullptr);
			break;
		default:
			errno = EINVAL;
		}

		if (Handle != INVALID_HANDLE_VALUE)
		{
			if (!(mode & 4) && (BiasStart) && (BiasLength != -1))
			{
				Seek(0,0);
			}
			return true;
		}
		return false;
	}


	int Read(void* buffer, int bytesToRead)
	{
		DWORD bytesRead = 0;
		if (!Is_Open())
		{
			if (Open(1))
			{
				bytesRead = Read(buffer, bytesToRead);
				Close();
			}
		}
		else
		{
			if (BiasLength != -1)
			{
				int maxBytesToRead = BiasLength - Tell();
				if (bytesToRead > maxBytesToRead)
					bytesToRead = maxBytesToRead;
			}
			
			if (!ReadFile(Handle, buffer, bytesToRead, &bytesRead, nullptr))
				Error(GetLastError(), 1, Filename);
		}
		
		return bytesRead;
	}


	int Seek(int pos, int dir)
	{
		if (BiasLength != -1)
		{
			switch (dir)
			{
			case 1: pos += Raw_Seek(0, 1) - BiasStart; break;
			case 2: pos += BiasLength; break;
			}

			int result = Raw_Seek(BiasStart + clamp(pos, 0, BiasLength), 0);
			if (result != -1)
				result -= BiasStart;

			TT_ASSERT(result <= BiasLength);
			return result;
		}
		else
			return Raw_Seek(pos, dir);
	}


	int Raw_Seek(int pos, int dir)
	{
		if (!Is_Open())
			Error(9, 0, Filename);
		
		int seek = SetFilePointer(Handle, pos, nullptr, dir);
		if (seek == -1)
			Error(GetLastError(), 0, Filename);
		
		return seek;
	}


	virtual int Size() override
	{
		if (BiasLength != -1)
			return BiasLength;
		else
		{
			int size = -1;
			if (!Is_Open())
			{
				if (Open(1))
				{
					size = Size();
					Close();
				}
			}
			else
			{
				size = GetFileSize(Handle,nullptr);
				if (size == -1)
					Error(GetLastError(), 0, Filename);
			}
			return size;
		}
	}


	int Write(const void* buffer, int size)
	{
		DWORD bytesWritten = 0;
		if (!Is_Open())
		{
			if (Open(2))
			{
				bytesWritten = Write(buffer, size);
				Close();
			}
		}
		else
		{
			if (!WriteFile(Handle, buffer, size, &bytesWritten, nullptr))
				Error(GetLastError(), 0, Filename);
			
			if (BiasLength != -1)
			{
				if (Raw_Seek(0, 1) > BiasStart + BiasLength)
				{
					BiasLength = Raw_Seek(0, 1) - BiasStart;
				}
			}
		}
		return bytesWritten;
	}


	void Close()
	{
		if (Is_Open())
		{
			if (!CloseHandle(Handle))
				Error(GetLastError(), 0, Filename);
			
			Handle = INVALID_HANDLE_VALUE;
		}
	}


	unsigned long Get_Date_Time()
	{
		BY_HANDLE_FILE_INFORMATION info;
		unsigned short dostime;
		unsigned short dosdate;
		if (GetFileInformationByHandle(Handle,&info))
		{
			FileTimeToDosDateTime(&info.ftLastWriteTime, &dosdate, &dostime);
			return dosdate << 0x10 | dostime;
		}
		return 0;
	}
	bool Set_Date_Time(unsigned long datetime)
	{
		BY_HANDLE_FILE_INFORMATION info;
		FILETIME filetime;
		if (Is_Open())
		{
			if (GetFileInformationByHandle(Handle,&info))
			{
				if (DosDateTimeToFileTime((WORD)(datetime >> 0x10),(WORD)datetime,&filetime))
				{
					if (SetFileTime(Handle,&info.ftCreationTime,&filetime,&filetime))
					{
						return true;
					}
				}
			}
		}
		return false;
	}
	void Error(int a, int b, const char *c)
	{
	}

	virtual HANDLE Get_File_Handle() const override
	{
		return Handle;
	}

	void Bias(int start, int length)
	{
		if (start == 0)
		{
			TT_ASSERT(length == -1);
			BiasStart = 0;
			BiasLength = -1;
		}
		else
		{
			BiasLength = Size();
			BiasStart += start;
			if (length != -1)
			{
				if (length < BiasLength)
					BiasLength = length;
				
				if (BiasLength < 0)
					BiasLength = 0;
			}
		}
	}
    void Get_Bias(int& start, int& length) override
    { 
        start = BiasStart;
        length = BiasLength;
    }
	virtual void Attach(HANDLE handle,int rights)
	{
		Reset();
		Handle = handle;
		Rights = rights;
		BiasStart = 0;
		BiasLength = -1;
		Date = 0;
		Time = 0;
	}
	virtual void Detach()
	{
		Rights = 0;
		Handle = INVALID_HANDLE_VALUE;
		BiasStart = 0;
		BiasLength = -1;
		Date = 0;
		Time = 0;
	}
}; 

class TextFileClass : public RawFileClass {
public:
	TextFileClass()
	{
	}
	TextFileClass(const char *filename) : RawFileClass(filename)
	{
	}
	~TextFileClass()
	{
	}
	bool Read_Line(StringClass &str)
	{
		str = "";
		char buf[64];
		memset(buf,0,sizeof(buf));
		bool b;
		do
		{
			int sz = Read(buf,63);
			b = (sz == 63);
			if (sz > 0)
			{
				for (int i = 0;i < sz;i++)
				{
					if (buf[i] == '\n')
					{
						buf[i + 1] = 0;
						Seek(i - sz + 1,1);
						b = false;
						break;
					}
				}
				str += buf;
			}
		} while (b);
		bool l = str.Get_Length() > 0;
		strtrim(str.Peek_Buffer());
		return l;
	}
	bool Write_Line(StringClass const &str)
	{
		int len = str.Get_Length();
		if (Write((void *)str.Peek_Buffer(),len) == len)
		{
			return Write("\r\n",2) == 2;
		}
		return false;
	}
};

class WideTextFileClass : public RawFileClass {
public:
	WideTextFileClass()
	{
	}
	WideTextFileClass(const char *filename) : RawFileClass(filename)
	{
	}
	~WideTextFileClass()
	{
	}
	bool Read_Line(WideStringClass &wstr)
	{
		StringClass str = "";
		char buf[64];
		memset(buf, 0, sizeof(buf));
		bool b;
		do
		{
			int sz = Read(buf, 63);
			b = (sz == 63);
			if (sz > 0)
			{
				for (int i = 0; i < sz; i++)
				{
					if (buf[i] == '\n')
					{
						buf[i + 1] = 0;
						Seek(i - sz + 1, 1);
						b = false;
						break;
					}
				}
				str += buf;
			}
		} while (b);
		wstr = FromUTF8(str);
		if (wstr[wstr.Get_Length() - 1] == L'\n')
		{
			wstr.TruncateRight(1);
		}
		if (wstr[wstr.Get_Length() - 1] == L'\r')
		{
			wstr.TruncateRight(1);
		}
		bool l = wstr.Get_Length() > 0;
		return l;
	}
	bool Write_Line(WideStringClass const &str)
	{
		StringClass s = ToUTF8(str);
		int len = s.Get_Length();
		if (Write((void *)s.Peek_Buffer(), len) == len)
		{
			static StringClass s2 = ToUTF8(L"\r\n");
			int len2 = s2.Get_Length();
			return Write((void *)s2.Peek_Buffer(), len2) == len2;
		}
		return false;
	}
	StringClass ToUTF8(WideStringClass const &wstr)
	{
		if (wstr.Is_Empty())
		{
			return "";
		}
		int size = WideCharToMultiByte(CP_UTF8, 0, wstr.Peek_Buffer(), -1, nullptr, 0, nullptr, nullptr);
		StringClass str;
		WideCharToMultiByte(CP_UTF8, 0, wstr.Peek_Buffer(), -1, str.Get_Buffer(size), size, nullptr, nullptr);
		return str;
	}
	WideStringClass FromUTF8(StringClass const &str)
	{
		if (str.Is_Empty())
		{
			return "";
		}
		int size = MultiByteToWideChar(CP_UTF8, 0, str.Peek_Buffer(), -1, nullptr, 0);
		WideStringClass wstr;
		MultiByteToWideChar(CP_UTF8, 0, str.Peek_Buffer(), -1, wstr.Get_Buffer(size), size);
		return wstr;
	}
};

#endif
